/*
 *
Copyright 1990, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.
 *
 * Author:  Keith Packard, MIT X Consortium
 */


/*
 * choose.c
 *
 * xdm interface to chooser program
 */

#include "dm.h"
#include "dm_error.h"

#ifdef XDMCP

# include <X11/X.h>
# include <sys/types.h>

# include "dm_socket.h"
# include <arpa/inet.h>
# include <sys/un.h>

# include <ctype.h>
# include <errno.h>

# ifdef HAVE_SETPROCTITLE
#  include <unistd.h>
# endif

# include <time.h>
# define Time_t time_t

static int
FormatBytes (
    unsigned char *data,
    int	    length,
    char    *buf,
    int	    buflen)
{
    int	    i;
    static char	    HexChars[] = "0123456789abcdef";

    if (buflen < length * 2 + 1)
	return 0;
    for (i = 0; i < length; i++)
    {
	*buf++ = HexChars[(data[i] >> 4) & 0xf];
	*buf++ = HexChars[(data[i]) & 0xf];
    }
    *buf++ = '\0';
    return 1;
}

static int
FormatARRAY8 (
    ARRAY8Ptr	a,
    char	*buf,
    int		buflen)
{
    return FormatBytes (a->data, a->length, buf, buflen);
}

/* Converts an Internet address in ARRAY8 format to a string in
   familiar dotted address notation, e.g., "18.24.0.11"
   Returns 1 if successful, 0 if not.
   */
static int
ARRAY8ToDottedDecimal (
    ARRAY8Ptr	a,
    char	*buf,
    int		buflen)
{
    int outlen;
    if (a->length != 4)
	return 0;
    outlen = snprintf(buf, buflen, "%d.%d.%d.%d",
		      a->data[0], a->data[1], a->data[2], a->data[3]);
    if (outlen >= buflen) {
	return 0;
    }
    return 1;
}

typedef struct _IndirectUsers {
    struct _IndirectUsers   *next;
    ARRAY8	client;
    CARD16	connectionType;
} IndirectUsersRec, *IndirectUsersPtr;

static IndirectUsersPtr	indirectUsers;

int
RememberIndirectClient (
    ARRAY8Ptr	clientAddress,
    CARD16	connectionType)
{
    IndirectUsersPtr	i;

    for (i = indirectUsers; i; i = i->next)
	if (XdmcpARRAY8Equal (clientAddress, &i->client) &&
	    connectionType == i->connectionType)
	    return 1;
    i = malloc (sizeof (IndirectUsersRec));
    if (!i)
    {
	LogOutOfMem ("RememberIndirectClient\n");
	return 0;
    }
    if (!XdmcpCopyARRAY8 (clientAddress, &i->client))
    {
	free (i);
	return 0;
    }
    i->connectionType = connectionType;
    i->next = indirectUsers;
    indirectUsers = i;
    return 1;
}

void
ForgetIndirectClient (
    ARRAY8Ptr	clientAddress,
    CARD16	connectionType)
{
    IndirectUsersPtr	i, prev;

    prev = NULL;
    for (i = indirectUsers; i; i = i->next)
    {
	if (XdmcpARRAY8Equal (clientAddress, &i->client) &&
	    connectionType == i->connectionType)
	{
	    if (prev)
		prev->next = i->next;
	    else
		indirectUsers = i->next;
	    XdmcpDisposeARRAY8 (&i->client);
	    free (i);
	    break;
	}
	prev = i;
    }
}

int
IsIndirectClient (
    ARRAY8Ptr	clientAddress,
    CARD16	connectionType)
{
    IndirectUsersPtr	i;

    for (i = indirectUsers; i; i = i->next)
	if (XdmcpARRAY8Equal (clientAddress, &i->client) &&
	    connectionType == i->connectionType)
	    return 1;
    return 0;
}

static int
FormatChooserArgument (char *buf, int len)
{
    unsigned char   addr_buf[1024];
    int		    addr_len = sizeof (addr_buf);
    unsigned char   result_buf[1024];
    int		    result_len = 0;
    int		    netfamily;

    if (GetChooserAddr ((char *)addr_buf, &addr_len) == -1)
    {
	LogError ("Cannot get return address for chooser socket\n");
	Debug ("Cannot get chooser socket address\n");
	return 0;
    }
    netfamily = NetaddrFamily((XdmcpNetaddr)addr_buf);
    switch (netfamily) {
    case AF_INET:
# ifdef IPv6
    case AF_INET6:
# endif
	{
	    char *port;
	    int portlen;
	    ARRAY8Ptr localAddress = getLocalAddress ();

# ifdef IPv6
	    if (localAddress->length == 16)
		netfamily = AF_INET6;
	    else
		netfamily = AF_INET;
# endif

	    port = NetaddrPort((XdmcpNetaddr)addr_buf, &portlen);
	    if (port == NULL) {
		LogError ("Cannot get port for chooser socket\n");
		return 0;
	    }
	    result_buf[0] = netfamily >> 8;
	    result_buf[1] = netfamily & 0xFF;
	    result_buf[2] = port[0];
	    result_buf[3] = port[1];
	    memcpy(result_buf+4, localAddress->data, localAddress->length);
	    result_len = 4 + localAddress->length;
	}
	break;
    default:
	Debug ("Chooser family %d isn't known\n", netfamily);
	return 0;
    }

    return FormatBytes (result_buf, result_len, buf, len);
}

typedef struct _Choices {
    struct _Choices *next;
    ARRAY8	    client;
    CARD16	    connectionType;
    ARRAY8	    choice;
    Time_t	    time;
} ChoiceRec, *ChoicePtr;

static ChoicePtr   choices;

ARRAY8Ptr
IndirectChoice (
    ARRAY8Ptr	clientAddress,
    CARD16	connectionType)
{
    ChoicePtr	c, next, prev;
    Time_t	now;

    now = time ((Time_t*)0);
    prev = NULL;
    for (c = choices; c; c = next)
    {
	next = c->next;
	Debug ("Choice checking timeout: %ld >? %d\n",
	    (long)(now - c->time), choiceTimeout);
	if (now - c->time > (Time_t)choiceTimeout)
	{
	    Debug ("Timeout choice %ld > %d\n",
		(long)(now - c->time), choiceTimeout);
	    if (prev)
		prev->next = next;
	    else
		choices = next;
	    XdmcpDisposeARRAY8 (&c->client);
	    XdmcpDisposeARRAY8 (&c->choice);
	    free (c);
	}
	else
	{
	    if (XdmcpARRAY8Equal (clientAddress, &c->client) &&
		connectionType == c->connectionType)
		return &c->choice;
	    prev = c;
	}
    }
    return NULL;
}

static int
RegisterIndirectChoice (
    ARRAY8Ptr	clientAddress,
    CARD16      connectionType,
    ARRAY8Ptr	choice)
{
    ChoicePtr	c;
    int		insert;

    Debug ("Got indirect choice back\n");
    for (c = choices; c; c = c->next) {
	if (XdmcpARRAY8Equal (clientAddress, &c->client) &&
	    connectionType == c->connectionType) {
	    break;
	}
    }
    insert = 0;
    if (!c)
    {
	c = malloc (sizeof (ChoiceRec));
	insert = 1;
	if (!c)
	    return 0;
	c->connectionType = connectionType;
	if (!XdmcpCopyARRAY8 (clientAddress, &c->client))
	{
	    free (c);
	    return 0;
	}
    }
    else
    {
	XdmcpDisposeARRAY8 (&c->choice);
    }
    if (!XdmcpCopyARRAY8 (choice, &c->choice))
    {
	XdmcpDisposeARRAY8 (&c->client);
	free (c);
	return 0;
    }
    if (insert)
    {
	c->next = choices;
	choices = c;
    }
    c->time = time ((Time_t *) 0);
    return 1;
}

# ifdef notdef
static
RemoveIndirectChoice (clientAddress, connectionType)
    ARRAY8Ptr	clientAddress;
    CARD16	connectionType;
{
    ChoicePtr	c, prev;

    prev = 0;
    for (c = choices; c; c = c->next)
    {
	if (XdmcpARRAY8Equal (clientAddress, &c->client) &&
	    connectionType == c->connectionType)
	{
	    if (prev)
		prev->next = c->next;
	    else
		choices = c->next;
	    XdmcpDisposeARRAY8 (&c->client);
	    XdmcpDisposeARRAY8 (&c->choice);
	    free (c);
	    return;
	}
	prev = c;
    }
}
# endif

/*ARGSUSED*/
static void
AddChooserHost (
    CARD16	connectionType,
    ARRAY8Ptr	addr,
    char	*closure)
{
    char	***argp;
    char	hostbuf[1024];

    argp = (char ***) closure;
    if (addr->length == strlen ("BROADCAST") &&
	!strncmp ((char *)addr->data, "BROADCAST", addr->length))
    {
	*argp = parseArgs (*argp, "BROADCAST");
    }
# ifdef IPv6
    else if ( (addr->length == 16) &&
      (inet_ntop(AF_INET6, addr->data, hostbuf, sizeof(hostbuf))))
    {
	*argp = parseArgs (*argp, hostbuf);
    }
# endif
    else if (ARRAY8ToDottedDecimal (addr, hostbuf, sizeof (hostbuf)))
    {
	*argp = parseArgs (*argp, hostbuf);
    }
}

void
ProcessChooserSocket (int fd)
{
    int client_fd;
    char	buf[1024];
    int		len;
    XdmcpBuffer	buffer;
    ARRAY8	clientAddress = {0, NULL};
    CARD16	connectionType;
    ARRAY8	choice = {0, NULL};

    Debug ("Process chooser socket\n");
    len = sizeof (buf);
    client_fd = accept (fd, (struct sockaddr *)buf, (void *)&len);
    if (client_fd == -1)
    {
	LogError ("Cannot accept chooser connection\n");
	return;
    }
    Debug ("Accepted %d\n", client_fd);

    len = read (client_fd, buf, sizeof (buf));
    Debug ("Read returns %d\n", len);
    if (len > 0)
    {
	buffer.data = (BYTE *) buf;
	buffer.size = sizeof (buf);
	buffer.count = len;
	buffer.pointer = 0;
	if (XdmcpReadARRAY8 (&buffer, &clientAddress)) {
	    if (XdmcpReadCARD16 (&buffer, &connectionType)) {
		if (XdmcpReadARRAY8 (&buffer, &choice)) {
		    Debug ("Read from chooser successfully\n");
		    RegisterIndirectChoice (&clientAddress, connectionType, &choice);
		    XdmcpDisposeARRAY8 (&choice);
		} else {
		    LogError ("Invalid choice response length %d\n", len);
		}
	    } else {
		LogError ("Invalid choice response length %d\n", len);
	    }
	    XdmcpDisposeARRAY8 (&clientAddress);
	} else {
	    LogError ("Invalid choice response length %d\n", len);
	}
    }
    else
    {
	LogError ("Choice response read error: %s\n", _SysErrorMsg(errno));
    }

    close (client_fd);
}

void
RunChooser (struct display *d)
{
    char    **args;
    char    buf[1024];
    char    **env;

    Debug ("RunChooser %s\n", d->name);
# ifndef HAVE_SETPROCTITLE
    SetTitle (d->name, "chooser", (char *) 0);
# else
    setproctitle("chooser %s", d->name);
# endif
    LoadXloginResources (d);
    args = parseArgs ((char **) 0, d->chooser);
    strcpy (buf, "-xdmaddress ");
    if (FormatChooserArgument (buf + strlen (buf), sizeof (buf) - strlen (buf)))
	args = parseArgs (args, buf);
    strcpy (buf, "-clientaddress ");
    if (FormatARRAY8 (&d->clientAddr, buf + strlen (buf), sizeof (buf) - strlen (buf)))
	args = parseArgs (args, buf);
    snprintf (buf, sizeof(buf), "-connectionType %d", d->connectionType);
    args = parseArgs (args, buf);
    ForEachChooserHost (&d->clientAddr, d->connectionType, AddChooserHost,
			(char *) &args);
    env = systemEnv (d, (char *) 0, (char *) 0);
    Debug ("Running %s\n", args[0]);
    execute (args, env);
    Debug ("Couldn't run %s\n", args[0]);
    LogError ("Cannot execute %s\n", args[0]);
    exit (REMANAGE_DISPLAY);
}

#endif /* XDMCP */
